// TcpBridge2IO.as - Bridge SimulIDE <-> Godot por TCP
// Versão para SimulIDE 2.0 (tester R250808)
//
// Entradas da planta (saídas do controlador, pinos de ENTRADA do bloco):
//   Esteira 1:      %QX0.0 .. %QX0.7, %QW0 .. %QW3
//   Abastecimento:  %QX1.0 .. %QX1.6
//   Nível simples:  %QX2.0, %QX2.1, %QX2.2
//   Portão:         %QX3.0, %QX3.1
//
// Saídas da planta (entradas do controlador, pinos de SAÍDA do bloco):
//   Esteira 1:      %IX0.0 .. %IX0.10
//   Abastecimento:  %IX1.0 .. %IX1.8
//   Nível simples:  %IX2.0, %IX2.1, %IW2 (analógico)
//   Portão:         %IX3.0 .. %IX3.3, %IX3.5 .. %IX3.8
//
// Analógicos:
//   %QW0..%QW3: 0–1023 <-> 0–5 V (entrada planta)
//   %IW2:       0–1023 <-> 0–5 V (saída planta)
//
// SimulIDE -> Godot:  chaves %QX*, %QW*
// Godot   -> SimulIDE: chaves %IX*, %IW2

// --------------------------------------------------------
// Pinos
// --------------------------------------------------------

// Entradas planta - Esteira 1
IoPin@ pinQX0_0;
IoPin@ pinQX0_1;
IoPin@ pinQX0_2;
IoPin@ pinQX0_3;
IoPin@ pinQX0_4;
IoPin@ pinQX0_5;
IoPin@ pinQX0_6;
IoPin@ pinQX0_7;
IoPin@ pinQW0;
IoPin@ pinQW1;
IoPin@ pinQW2;
IoPin@ pinQW3;

// Entradas planta - Abastecimento de água
IoPin@ pinQX1_0;
IoPin@ pinQX1_1;
IoPin@ pinQX1_2;
IoPin@ pinQX1_3;
IoPin@ pinQX1_4;
IoPin@ pinQX1_5;
IoPin@ pinQX1_6;

// Entradas planta - Nível simples
IoPin@ pinQX2_0;
IoPin@ pinQX2_1;
IoPin@ pinQX2_2;

// Entradas planta - Portão
IoPin@ pinQX3_0;
IoPin@ pinQX3_1;

// Saídas planta - Esteira 1
IoPin@ pinIX0_0;
IoPin@ pinIX0_1;
IoPin@ pinIX0_2;
IoPin@ pinIX0_3;
IoPin@ pinIX0_4;
IoPin@ pinIX0_5;
IoPin@ pinIX0_6;
IoPin@ pinIX0_7;
IoPin@ pinIX0_8;
IoPin@ pinIX0_9;
IoPin@ pinIX0_10;

// Saídas planta - Abastecimento de água
IoPin@ pinIX1_0;
IoPin@ pinIX1_1;
IoPin@ pinIX1_2;
IoPin@ pinIX1_3;
IoPin@ pinIX1_4;
IoPin@ pinIX1_5;
IoPin@ pinIX1_6;
IoPin@ pinIX1_7;
IoPin@ pinIX1_8;

// Saídas planta - Nível simples
IoPin@ pinIX2_0;
IoPin@ pinIX2_1;
IoPin@ pinIW2;

// Saídas planta - Portão
IoPin@ pinIX3_0;
IoPin@ pinIX3_1;
IoPin@ pinIX3_2;
IoPin@ pinIX3_3;
IoPin@ pinIX3_5;
IoPin@ pinIX3_6;
IoPin@ pinIX3_7;
IoPin@ pinIX3_8;

// Estado TCP
bool m_connected = false;
int  send_counter = 0;

// JSON builder global (sem &inout)
string g_json;
bool   g_json_first;

// --------------------------------------------------------
// Helpers JSON (usam globais)
// --------------------------------------------------------
void jsonBegin()
{
    g_json = "{";
    g_json_first = true;
}

void jsonAppendBool(const string &in key, bool val)
{
    if (!g_json_first) g_json += ",";
    g_json_first = false;
    g_json += "\"";
    g_json += key;
    g_json += "\":";
    g_json += val ? "true" : "false";
}

void jsonAppendInt(const string &in key, int val)
{
    if (!g_json_first) g_json += ",";
    g_json_first = false;
    g_json += "\"";
    g_json += key;
    g_json += "\":";
    g_json += "" + val;
}

string jsonEnd()
{
    g_json += "}";
    return g_json;
}

// --------------------------------------------------------
// setup(): mapeia pinos
// --------------------------------------------------------
void setup()
{
    // Entradas planta - Esteira 1
    @pinQX0_0 = component.getPin("%QX0.0");
    @pinQX0_1 = component.getPin("%QX0.1");
    @pinQX0_2 = component.getPin("%QX0.2");
    @pinQX0_3 = component.getPin("%QX0.3");
    @pinQX0_4 = component.getPin("%QX0.4");
    @pinQX0_5 = component.getPin("%QX0.5");
    @pinQX0_6 = component.getPin("%QX0.6");
    @pinQX0_7 = component.getPin("%QX0.7");
    @pinQW0   = component.getPin("%QW0");
    @pinQW1   = component.getPin("%QW1");
    @pinQW2   = component.getPin("%QW2");
    @pinQW3   = component.getPin("%QW3");

    // Entradas planta - Abastecimento
    @pinQX1_0 = component.getPin("%QX1.0");
    @pinQX1_1 = component.getPin("%QX1.1");
    @pinQX1_2 = component.getPin("%QX1.2");
    @pinQX1_3 = component.getPin("%QX1.3");
    @pinQX1_4 = component.getPin("%QX1.4");
    @pinQX1_5 = component.getPin("%QX1.5");
    @pinQX1_6 = component.getPin("%QX1.6");

    // Entradas planta - Nível simples
    @pinQX2_0 = component.getPin("%QX2.0");
    @pinQX2_1 = component.getPin("%QX2.1");
    @pinQX2_2 = component.getPin("%QX2.2");

    // Entradas planta - Portão
    @pinQX3_0 = component.getPin("%QX3.0");
    @pinQX3_1 = component.getPin("%QX3.1");

    // Saídas planta - Esteira 1
    @pinIX0_0  = component.getPin("%IX0.0");
    @pinIX0_1  = component.getPin("%IX0.1");
    @pinIX0_2  = component.getPin("%IX0.2");
    @pinIX0_3  = component.getPin("%IX0.3");
    @pinIX0_4  = component.getPin("%IX0.4");
    @pinIX0_5  = component.getPin("%IX0.5");
    @pinIX0_6  = component.getPin("%IX0.6");
    @pinIX0_7  = component.getPin("%IX0.7");
    @pinIX0_8  = component.getPin("%IX0.8");
    @pinIX0_9  = component.getPin("%IX0.9");
    @pinIX0_10 = component.getPin("%IX0.10");

    // Saídas planta - Abastecimento
    @pinIX1_0 = component.getPin("%IX1.0");
    @pinIX1_1 = component.getPin("%IX1.1");
    @pinIX1_2 = component.getPin("%IX1.2");
    @pinIX1_3 = component.getPin("%IX1.3");
    @pinIX1_4 = component.getPin("%IX1.4");
    @pinIX1_5 = component.getPin("%IX1.5");
    @pinIX1_6 = component.getPin("%IX1.6");
    @pinIX1_7 = component.getPin("%IX1.7");
    @pinIX1_8 = component.getPin("%IX1.8");

    // Saídas planta - Nível simples
    @pinIX2_0 = component.getPin("%IX2.0");
    @pinIX2_1 = component.getPin("%IX2.1");
    @pinIW2   = component.getPin("%IW2");

    // Saídas planta - Portão
    @pinIX3_0 = component.getPin("%IX3.0");
    @pinIX3_1 = component.getPin("%IX3.1");
    @pinIX3_2 = component.getPin("%IX3.2");
    @pinIX3_3 = component.getPin("%IX3.3");
    @pinIX3_5 = component.getPin("%IX3.5");
    @pinIX3_6 = component.getPin("%IX3.6");
    @pinIX3_7 = component.getPin("%IX3.7");
    @pinIX3_8 = component.getPin("%IX3.8");

    print("TcpBridge2IO: setup()\n");
}

// --------------------------------------------------------
// reset(): configura modos e conecta TCP
// --------------------------------------------------------
void reset()
{
    print("TcpBridge2IO: reset()\n");

    m_connected  = false;
    send_counter = 0;

    // Saídas planta -> OUTPUT (0 V)
    setOutputPin(pinIX0_0);
    setOutputPin(pinIX0_1);
    setOutputPin(pinIX0_2);
    setOutputPin(pinIX0_3);
    setOutputPin(pinIX0_4);
    setOutputPin(pinIX0_5);
    setOutputPin(pinIX0_6);
    setOutputPin(pinIX0_7);
    setOutputPin(pinIX0_8);
    setOutputPin(pinIX0_9);
    setOutputPin(pinIX0_10);

    setOutputPin(pinIX1_0);
    setOutputPin(pinIX1_1);
    setOutputPin(pinIX1_2);
    setOutputPin(pinIX1_3);
    setOutputPin(pinIX1_4);
    setOutputPin(pinIX1_5);
    setOutputPin(pinIX1_6);
    setOutputPin(pinIX1_7);
    setOutputPin(pinIX1_8);

    setOutputPin(pinIX2_0);
    setOutputPin(pinIX2_1);
    setOutputPin(pinIW2);

    setOutputPin(pinIX3_0);
    setOutputPin(pinIX3_1);
    setOutputPin(pinIX3_2);
    setOutputPin(pinIX3_3);
    setOutputPin(pinIX3_5);
    setOutputPin(pinIX3_6);
    setOutputPin(pinIX3_7);
    setOutputPin(pinIX3_8);

    // Entradas planta -> INPUT
    setInputPin(pinQX0_0);
    setInputPin(pinQX0_1);
    setInputPin(pinQX0_2);
    setInputPin(pinQX0_3);
    setInputPin(pinQX0_4);
    setInputPin(pinQX0_5);
    setInputPin(pinQX0_6);
    setInputPin(pinQX0_7);
    setInputPin(pinQW0);
    setInputPin(pinQW1);
    setInputPin(pinQW2);
    setInputPin(pinQW3);

    setInputPin(pinQX1_0);
    setInputPin(pinQX1_1);
    setInputPin(pinQX1_2);
    setInputPin(pinQX1_3);
    setInputPin(pinQX1_4);
    setInputPin(pinQX1_5);
    setInputPin(pinQX1_6);

    setInputPin(pinQX2_0);
    setInputPin(pinQX2_1);
    setInputPin(pinQX2_2);

    setInputPin(pinQX3_0);
    setInputPin(pinQX3_1);

    // Conecta TCP
    Wb_link.connectToHost(0, "127.0.0.1", 8765);
}

void setOutputPin(IoPin@ p)
{
    if (p is null) return;
    p.setPinMode(3);      // output
    p.setOutHighV(5.0);
    p.setVoltage(0.0);
}

void setInputPin(IoPin@ p)
{
    if (p is null) return;
    p.setPinMode(1);      // input
}

// --------------------------------------------------------
// updateStep(): envia %QX* e %QW* para o Godot
// --------------------------------------------------------
void updateStep()
{
    if (!m_connected) return;

    send_counter++;
    if (send_counter < 2) return;
    send_counter = 0;

    jsonBegin();

    // Esteira 1 - digitais
    jsonAppendBool("%QX0.0", getBool(pinQX0_0));
    jsonAppendBool("%QX0.1", getBool(pinQX0_1));
    jsonAppendBool("%QX0.2", getBool(pinQX0_2));
    jsonAppendBool("%QX0.3", getBool(pinQX0_3));
    jsonAppendBool("%QX0.4", getBool(pinQX0_4));
    jsonAppendBool("%QX0.5", getBool(pinQX0_5));
    jsonAppendBool("%QX0.6", getBool(pinQX0_6));
    jsonAppendBool("%QX0.7", getBool(pinQX0_7));

    // Esteira 1 - analógicos
    jsonAppendInt("%QW0", analogToInt(pinQW0));
    jsonAppendInt("%QW1", analogToInt(pinQW1));
    jsonAppendInt("%QW2", analogToInt(pinQW2));
    jsonAppendInt("%QW3", analogToInt(pinQW3));

    // Abastecimento
    jsonAppendBool("%QX1.0", getBool(pinQX1_0));
    jsonAppendBool("%QX1.1", getBool(pinQX1_1));
    jsonAppendBool("%QX1.2", getBool(pinQX1_2));
    jsonAppendBool("%QX1.3", getBool(pinQX1_3));
    jsonAppendBool("%QX1.4", getBool(pinQX1_4));
    jsonAppendBool("%QX1.5", getBool(pinQX1_5));
    jsonAppendBool("%QX1.6", getBool(pinQX1_6));

    // Nível simples
    jsonAppendBool("%QX2.0", getBool(pinQX2_0));
    jsonAppendBool("%QX2.1", getBool(pinQX2_1));
    jsonAppendBool("%QX2.2", getBool(pinQX2_2));

    // Portão
    jsonAppendBool("%QX3.0", getBool(pinQX3_0));
    jsonAppendBool("%QX3.1", getBool(pinQX3_1));

    string json = jsonEnd();
    Wb_link.sendMsgToHost(json + "\n", 0);
}

bool getBool(IoPin@ p)
{
    if (p is null) return false;
    return p.getInpState();
}

// Converte tensão 0–5 V em 0–1023
int analogToInt(IoPin@ p)
{
    if (p is null) return 0;
    double v = p.getVoltage();
    if (v < 0.0) v = 0.0;
    if (v > 5.0) v = 5.0;
    int val = int((v / 5.0) * 1023.0 + 0.5);
    return val;
}

// --------------------------------------------------------
// Callbacks TCP
// --------------------------------------------------------
void received(string msg, int link)
{
    parseOutputs(msg);
}

void tcpConnected(int link)
{
    print("TcpBridge2IO: tcpConnected link=" + link + "\n");
    m_connected = true;
}

void tcpDisconnected(int link)
{
    print("TcpBridge2IO: tcpDisconnected link=" + link + "\n");
    m_connected = false;

    // Saídas em estado seguro
    setVoltageSafe(pinIX0_0);
    setVoltageSafe(pinIX0_1);
    setVoltageSafe(pinIX0_2);
    setVoltageSafe(pinIX0_3);
    setVoltageSafe(pinIX0_4);
    setVoltageSafe(pinIX0_5);
    setVoltageSafe(pinIX0_6);
    setVoltageSafe(pinIX0_7);
    setVoltageSafe(pinIX0_8);
    setVoltageSafe(pinIX0_9);
    setVoltageSafe(pinIX0_10);

    setVoltageSafe(pinIX1_0);
    setVoltageSafe(pinIX1_1);
    setVoltageSafe(pinIX1_2);
    setVoltageSafe(pinIX1_3);
    setVoltageSafe(pinIX1_4);
    setVoltageSafe(pinIX1_5);
    setVoltageSafe(pinIX1_6);
    setVoltageSafe(pinIX1_7);
    setVoltageSafe(pinIX1_8);

    setVoltageSafe(pinIX2_0);
    setVoltageSafe(pinIX2_1);
    setVoltageSafe(pinIW2);

    setVoltageSafe(pinIX3_0);
    setVoltageSafe(pinIX3_1);
    setVoltageSafe(pinIX3_2);
    setVoltageSafe(pinIX3_3);
    setVoltageSafe(pinIX3_5);
    setVoltageSafe(pinIX3_6);
    setVoltageSafe(pinIX3_7);
    setVoltageSafe(pinIX3_8);
}

void setVoltageSafe(IoPin@ p)
{
    if (p is null) return;
    p.setVoltage(0.0);
}

void voltChanged()
{
    // não usado
}

// --------------------------------------------------------
// Parsing JSON Godot -> SimulIDE (%IX*, %IW2)
// --------------------------------------------------------
void parseOutputs(string json)
{
    // Esteira 1
    parseBoolKey(json, "%IX0.0",  pinIX0_0);
    parseBoolKey(json, "%IX0.1",  pinIX0_1);
    parseBoolKey(json, "%IX0.2",  pinIX0_2);
    parseBoolKey(json, "%IX0.3",  pinIX0_3);
    parseBoolKey(json, "%IX0.4",  pinIX0_4);
    parseBoolKey(json, "%IX0.5",  pinIX0_5);
    parseBoolKey(json, "%IX0.6",  pinIX0_6);
    parseBoolKey(json, "%IX0.7",  pinIX0_7);
    parseBoolKey(json, "%IX0.8",  pinIX0_8);
    parseBoolKey(json, "%IX0.9",  pinIX0_9);
    parseBoolKey(json, "%IX0.10", pinIX0_10);

    // Abastecimento
    parseBoolKey(json, "%IX1.0", pinIX1_0);
    parseBoolKey(json, "%IX1.1", pinIX1_1);
    parseBoolKey(json, "%IX1.2", pinIX1_2);
    parseBoolKey(json, "%IX1.3", pinIX1_3);
    parseBoolKey(json, "%IX1.4", pinIX1_4);
    parseBoolKey(json, "%IX1.5", pinIX1_5);
    parseBoolKey(json, "%IX1.6", pinIX1_6);
    parseBoolKey(json, "%IX1.7", pinIX1_7);
    parseBoolKey(json, "%IX1.8", pinIX1_8);

    // Nível simples
    parseBoolKey(json, "%IX2.0", pinIX2_0);
    parseBoolKey(json, "%IX2.1", pinIX2_1);
    parseAnalogKey(json, "%IW2", pinIW2);

    // Portão
    parseBoolKey(json, "%IX3.0", pinIX3_0);
    parseBoolKey(json, "%IX3.1", pinIX3_1);
    parseBoolKey(json, "%IX3.2", pinIX3_2);
    parseBoolKey(json, "%IX3.3", pinIX3_3);
    parseBoolKey(json, "%IX3.5", pinIX3_5);
    parseBoolKey(json, "%IX3.6", pinIX3_6);
    parseBoolKey(json, "%IX3.7", pinIX3_7);
    parseBoolKey(json, "%IX3.8", pinIX3_8);
}

// Booleano: "\"<key>\":true/false"
void parseBoolKey(string json, const string &in key, IoPin@ outPin)
{
    if (outPin is null) return;

    string pattern = "\"" + key + "\"";
    int posKey = json.findFirst(pattern);
    if (posKey < 0) return;

    int posColon = json.findFirst(":", posKey);
    if (posColon < 0) return;

    int posVal = posColon + 1;

    if (json.substr(posVal, 4) == "true")
    {
        outPin.setVoltage(5.0);
    }
    else if (json.substr(posVal, 5) == "false")
    {
        outPin.setVoltage(0.0);
    }
}

// Analógico: "\"<key>\":<numero>"  -> 0–1023 -> 0–5 V
void parseAnalogKey(string json, const string &in key, IoPin@ outPin)
{
    if (outPin is null) return;

    string pattern = "\"" + key + "\"";
    int posKey = json.findFirst(pattern);
    if (posKey < 0) return;

    int posColon = json.findFirst(":", posKey);
    if (posColon < 0) return;

    int posVal = posColon + 1;
    int len    = int(json.length());

    int value     = 0;
    bool hasDigit = false;

    for (int i = posVal; i < len; i++)
    {
        string ch = json.substr(i, 1);

        if (ch < "0" || ch > "9")
            break;

        hasDigit = true;

        int digit = 0;
        if      (ch == "1") digit = 1;
        else if (ch == "2") digit = 2;
        else if (ch == "3") digit = 3;
        else if (ch == "4") digit = 4;
        else if (ch == "5") digit = 5;
        else if (ch == "6") digit = 6;
        else if (ch == "7") digit = 7;
        else if (ch == "8") digit = 8;
        else if (ch == "9") digit = 9;
        // "0" => 0

        value = value * 10 + digit;
    }

    if (!hasDigit) return;

    if (value < 0) value = 0;
    if (value > 1023) value = 1023;

    double v = (double(value) / 1023.0) * 5.0;
    outPin.setVoltage(v);
}
